import React from 'react';
import classNames from 'classnames';

import MediaInsert from '../components/media-insert';
import MediaUpdate from '../components/media-update';
import DropZone from '../components/drop-zone';
import { MediaItem } from '../types/media';

type MediaUnitProps = {
  items: MediaItem[],
  mediaSizes: { value: string, label: string, selected: string }[],
  primaryImageId: string,
  id: string,
  bid: string,
  diff: string,
  active: string,
  mediaDir: string,
  rootDir: string,
  lang: string,
  enlarged: 'true' | 'false',
  primary: 'true' | 'false'
}

type MediaUnitState = {
  insertModalOpened: boolean,
  updateModalOpened: boolean,
  newMedia: boolean,
  noLarge: 'yes' | 'no',
  modalType: 'upload' | 'select',
  targetId: string,
  items: MediaItem[],
  files: File[],
  landscape: boolean
}

export default class MediaUnit extends React.Component<MediaUnitProps, MediaUnitState> {

  static defaultProps = {
    items: [],
    mediaSizes: [],
    lang: 'ja'
  }

  constructor(props: MediaUnitProps) {
    super(props);
    let newMedia = false;
    if (!props.items[0].media_id) {
      newMedia = true;
    }
    this.state = {
      insertModalOpened: false,
      updateModalOpened: false,
      modalType: 'upload',
      targetId: null,
      items: props.items,
      files: [],
      landscape: true,
      newMedia,
      noLarge: props.enlarged === "true" ? "no" : "yes" 
    };
  }

  componentDidMount() {
    const [item] = this.state.items;
    const src = item.media_thumbnail;
    const img = new Image();
    img.onload = () => {
      if (img.width < img.height) {
        this.setState({
          landscape: false
        });
      }
    }
    img.src = src;
  }

  openInsertModal = () => {
    this.setState({
      modalType: 'select',
      insertModalOpened: true
    });
  }

  openUploadModal = () => {
    this.setState({
      modalType: 'upload',
      insertModalOpened: true
    })
  }

  onInsert = (items) => {
    this.setState({
      items,
      insertModalOpened: false
    });
  }

  onClose = () => {
    this.setState({
      insertModalOpened: false
    });
  }

  onUpdateModalClose = () => {
    this.setState({
      updateModalOpened: false
    })
  }

  onUpdateModalUpdate = (item) => {
    const { items, targetId } = this.state;
    const findIndex = items.findIndex((currentItem) => {
      if (currentItem.media_id === targetId) {
        return true
      }
      return false;
    });
    const updated = [...items.slice(0, findIndex), item, ...items.slice(findIndex + 1)]
    this.setState({
      updateModalOpened: false,
      items: updated
    });
  }

  removeMediaAt = (index: number) => {
    const { items } = this.state;
    const filtered = items.filter((item, i) => i !== index);
    this.setState({
      items: filtered
    });
  }

  openEditModal = (targetId: string) => {
    this.setState({
      targetId,
      updateModalOpened: true
    });
  }

  onComplete = (files) => {
    this.setState({
      files: files.map(item => item.file),
      insertModalOpened: true,
      modalType: "upload"
    });
  }

  uploadFile = (e) => {
    this.setState({
      files: Array.from(e.target.files),
      insertModalOpened: true,
      modalType: "upload"
    });
  }

  render() {
    const { insertModalOpened, updateModalOpened, modalType, targetId, items, files, newMedia, noLarge } = this.state;
    const { mediaSizes, primaryImageId, id, active, rootDir, enlarged, primary, bid, lang } = this.props;
    const [item] = items;

    return (<DropZone onComplete={(files) => {
      this.onComplete(files);
    }} >
      <table className="entryFormColumnSettingTable entryFormColumnTable">
        <tr className="entryFormFileControl">
          <td className="acms-admin-media-unit-preview-area">
            {(items.length === 0 || (items.length < 2 && !item.media_id)) && <div>
              <div className="acms-admin-media-unit-droparea">
                <p className="acms-admin-media-unit-droparea-text">{ACMS.i18n("media.add_new_media")}</p>
                <label className="acms-admin-media-unit-droparea-btn" style={{ cursor: 'pointer' }}>{ACMS.i18n("media.upload")}
                  {!insertModalOpened && <input type="file" onChange={this.uploadFile} style={{ display: 'none' }} multiple />}
                </label>
                <p className="acms-admin-media-unit-droparea-text">{ACMS.i18n("media.drop_file")}</p>
              </div>
              <input type="hidden" name={`media_id_${id}[]`} value="" />
            </div>}
            {(items.length > 0 && items.length < 2 && item.media_id) &&
              <div className="acms-admin-media-unit-preview-wrap">
                <div className="acms-admin-media-unit-preview-overlay"></div>
                <button type="button" className="acms-admin-media-unit-preview-remove-btn" onClick={() => this.removeMediaAt(0)}></button>
                <div className="acms-admin-media-unit-preview-edit-overlay"></div>
                {(item.media_type === 'image' || item.media_type === 'svg') &&
                  <img className="acms-admin-media-unit-preview" src={`${item.media_thumbnail}`} alt="" />}
                {item.media_type == 'file' &&
                  <div className="acms-admin-media-unit-file-icon-wrap">
                    <img className="acms-admin-media-unit-file-icon" src={`${item.media_thumbnail}`} alt="" />
                    <p className="acms-admin-media-unit-file-caption">{item.media_title}</p>
                  </div>
                }
                <button type="button" className="acms-admin-media-edit-btn acms-admin-media-unit-preview-edit-btn" onClick={() => this.openEditModal(item.media_id)}>編集</button>
                <input type="hidden" name={`media_id_${id}[]`} value={item && item.media_id} />
              </div>}
          </td>

          <td className="entryFormFileControl">
            <table className="acms-admin-margin-bottom-mini" style={{ width: '100%' }}>
              <tr>
                <th>ID</th>
                <td>
                  <div style={{ display: 'inline-block' }}>
                    {(item && item.media_id) && <span className="acms-admin-label acms-admin-label-default acms-admin-margin-right-mini">{item.media_id}</span>}
                  </div>
                  <button type="button" className="acms-admin-btn-admin" onClick={this.openInsertModal}>{ACMS.i18n("media.select_from_media")}</button>
                  {active !== 'on' && <span style={{ color: 'red' }}>&nbsp;&nbsp;{ACMS.i18n("media.media_unit_use1")}<a href={`${rootDir}bid/${bid}/admin/config_function/`}>{ACMS.i18n("media.media_unit_use2")}</a>{ACMS.i18n("media.media_unit_use3")}</span>}
                </td>
              </tr>
              {(mediaSizes.length !== 0) && <tr>
                <th><label htmlFor={`unit-media-size-${id}`}>{ACMS.i18n("media.size")}</label></th>
                <td>
                  <select name={`media_size_${id}[]`} id={`unit-media-size-${id}`}>
                    {mediaSizes.map((mediaSize) => {
                      return (<option value={mediaSize.value} selected={!!mediaSize.selected}>{mediaSize.label}</option>)
                    })}
                  </select>
                </td>
              </tr>}
              {items.length < 2 && <tr>
                <th>{ACMS.i18n("media.caption")}</th>
                <td>
                  <textarea name={`media_caption_${id}[]`} rows={1} id={`unit-media-caption-${id}`} className="acms-admin-form-width-full">{item && item.media_caption}</textarea>
                </td>
              </tr>}
              {items.length < 2 && <tr>
                <th>{ACMS.i18n("media.alt")}</th>
                <td>
                  <textarea name={`media_alt_${id}[]`} rows={1} id={`unit-media-alt-text-${id}`} className="acms-admin-form-width-full">{item && item.media_alt}</textarea>
                </td>
              </tr>}
              {(items.length < 2 && item && item.media_type === 'image' && !newMedia) && <tr>
                <th>{ACMS.i18n("media.main_image")}</th>
                <td>
                  <div className="acms-admin-form-radio">
                    <input type="radio" name="primary_image" value={primaryImageId} id={`input-radio-image_primary_image_${id}_${lang}`} defaultChecked={primary === 'true'} />
                    <label htmlFor={`input-radio-image_primary_image_${id}_${lang}`}><i className="acms-admin-ico-radio"></i>{ACMS.i18n("media.set_as_main_image")}</label>
                  </div>
                </td>
              </tr>}
              {(items.length < 2 && item && item.media_type === 'image') && <tr>
                <th>{ACMS.i18n("media.large_display")}</th>
                <td>
                  <div className="acms-admin-form-checkbox">
                    <input type="hidden" name={`media_enlarged_${id}[]`} value={noLarge} />
                    <input 
                      type="checkbox" 
                      value="no" 
                      id={`input-checkbox-media_enlarged_${id}_${lang}`} 
                      defaultChecked={enlarged === 'true'} 
                      onChange={(e) => {
                        if (e.target.checked) {
                          this.setState({
                            noLarge: "no"
                          })
                        } else {
                          this.setState({
                            noLarge: "yes"
                          })
                        }
                      }}
                    />
                    <label htmlFor={`input-checkbox-media_enlarged_${id}_${lang}`}><i className="acms-admin-ico-checkbox"></i>{ACMS.i18n("media.not_large_display")}</label>
                  </div>
                </td>
              </tr>}
              {(items.length < 2 && item && item.media_text) && <tr>
                <th>{ACMS.i18n("media.title")}</th>
                <td>
                  <span className="js-media-text">{item && item.media_text}</span>
                </td>
              </tr>}
              {(items.length < 2 && item && item.media_id) && <tr>
                <th>{ACMS.i18n("media.link")}</th>
                <td>
                  <button type="button" className="acms-admin-btn-admin" onClick={() => this.openEditModal(item.media_id)}>{ACMS.i18n("media.edit")}</button>&nbsp;
                  {(item && item.media_link) && <span className="js-media-link"><a href={item && item.media_link} target="_blank" rel="noopener noreferrer">{item && item.media_link}</a></span>}
                </td>
              </tr>}
            </table>
          </td>
        </tr>
        {items.length >= 2 && <tr>
          <td colSpan={2}>
            <div className="acms-admin-media-unit-preview-group">
              {items.map((item, index) => {
                const [width, height] = item.media_size.split(' x ');
                const landscape = width > height;
                return (
                  <div className="acms-admin-media-unit-preview-wrap acms-admin-media-unit-preview-group-item">
                    <div className="acms-admin-media-unit-preview-group-item-inner">
                      <div className="acms-admin-media-unit-preview-edit-overlay"></div>
                      {(item.media_type === 'image' || item.media_type === 'svg') &&
                        <img className="acms-admin-media-unit-preview-group-img" src={`${item.media_thumbnail}`} alt="" />}
                      {item.media_type == 'file' &&
                        <div className="acms-admin-media-unit-file-icon-wrap">
                          <img className="acms-admin-media-unit-file-icon" src={`${item.media_thumbnail}`} alt="" />
                          <p className="acms-admin-media-unit-file-caption">{item.media_title}</p>
                        </div>
                      }
                      <input type="hidden" name={`media_id_${id}[]`} value={item && item.media_id} />
                    </div>
                    <div className="acms-admin-media-unit-preview-overlay"></div>
                    <button type="button" className="acms-admin-media-unit-preview-remove-btn" onClick={() => this.removeMediaAt(index)}></button>
                    <button type="button" className="acms-admin-media-edit-btn acms-admin-media-unit-preview-edit-btn" onClick={() => this.openEditModal(item.media_id)}>{ACMS.i18n("media.edit")}</button>
                  </div>
                )
              })}
            </div>
          </td>
        </tr>}
      </table>
      {insertModalOpened && <MediaInsert
        onInsert={this.onInsert}
        {...(files.length ? { files } : {})}
        onClose={this.onClose}
        tab={modalType}
        filetype="all" />
      }
      {updateModalOpened && <MediaUpdate
        mid={targetId}
        onClose={this.onUpdateModalClose}
        onUpdate={this.onUpdateModalUpdate}
      />}
    </DropZone>)
  }
}
