import DocumentOutliner from 'document-outliner';
import tooltip from './lib/tooltip';
import ResizeImage from './lib/resize-image/resize-image';
import { findAncestor } from './lib/dom';
import { contrastColor, rgb2hex } from './lib/utility';
import { FocusedImage } from 'image-focus';
import ScrollHint from 'scroll-hint';

export default (context) => {
  //--------
  // resize
  $(ACMS.Config.resizeImageTargetMarkCF, context).each(function () {
    const resizeImg = new ResizeImage(this);
    resizeImg.resize();
  });

  const TableUnitScrollHint = '.js-table-unit-scroll-hint';

  if (document.querySelector(ACMS.Config.scrollHintMark) || document.querySelector(TableUnitScrollHint)) {
    //-------------
    // scroll hint
    import(/* webpackChunkName: "scroll-hint-css" */'scroll-hint/css/scroll-hint.css').then(() => {
      // build in js
      new ScrollHint(ACMS.Config.scrollHintMark, ACMS.Config.scrollHintConfig); // eslint-disable-line no-new

      // table unit
      new ScrollHint(TableUnitScrollHint, { ...ACMS.Config.scrollHintConfig, applyToParents: true }); // eslint-disable-line no-new
    });
  }

  //--------------
  // focus image
  [].forEach.call(document.querySelectorAll('.js-focused-image'), (image) => {
    image.style.visibility = 'visible';
    const newImage = new FocusedImage(image); // eslint-disable-line no-unused-vars
  });

  //---------------
  // OpenStreetMap
  [].forEach.call(document.querySelectorAll(ACMS.Config.openStreetMapMark), (item) => {
    import(/* webpackChunkName: "open-street-map" */'./lib/open-street-map').then(({ default: openStreetMap }) => {
      openStreetMap(item);
    });
  });

  //---------
  // tooltip
  $('.js-acms-tooltip', context).click(function () {
    tooltip(this);
    return false;
  });

  const tooltips = document.querySelectorAll('.js-acms-tooltip-hover');

  [].forEach.call(tooltips, (item) => {
    let interval;

    if (findAncestor(item, '#js-edit_inplace-box')) {
      return;
    }

    item.addEventListener('mouseenter', () => {
      tooltip(item, true);
      interval = setInterval(() => {
        if (!document.body.contains(item)) {
          $('.js-tooltip').remove();
          clearInterval(interval);
        }
      }, 300);
    });

    item.addEventListener('mouseleave', (e) => {
      clearInterval(interval);
      if ($(e.relatedTarget).hasClass('js-tooltip')) {
        const leaveFunc = (evt) => {
          if (evt.relatedTarget !== item) {
            evt.relatedTarget.removeEventListener('mouseleave', leaveFunc);
            tooltip(item, false);
          }
        };
        e.relatedTarget.addEventListener('mouseleave', leaveFunc);
      } else {
        tooltip(item, false);
      }
    });
  });

  //---------
  // preview
  ACMS.Dispatch.Preview = () => {
    import(/* webpackChunkName: "preview" */'./lib/preview').then(({ default: preview }) => {
      preview();
    });
  };
  if (window.parent !== window && location.href) {
    window.parent.postMessage({ task: 'preview', url: location.href }, '*');
  }

  $(ACMS.Config.externalFormSubmitButton).each((index, button) => {
    $(button).click((e) => {
      e.preventDefault();
      const target = $(button).data('target');
      if (!target) {
        return;
      }
      const name = $(button).attr('name');
      if (name && name.match(/^ACMS_POST/)) {
        $(target).append(`<input type="hidden" name="${name}" value="true" />`);
      }
      $(target).submit();
    });
  });

  $(ACMS.Config.blankSubmitBtn).each((index, button) => {
    const form = findAncestor(button, 'form');
    $(button).click(() => {
      $(form).attr('target', '_blank');
    });
    $(form).submit(() => {
      setTimeout(() => {
        $(form).removeAttr('target');
      }, 100);
    });
  });

  const dispatchFlatPicker = async (ctx) => {
    //-------------------
    // flatpicker
    const flatPickerTarget = ctx.querySelectorAll(ACMS.Config.flatDatePicker);
    if (flatPickerTarget && flatPickerTarget.length) {
      const { default: flatPicker } = await import(/* webpackChunkName: "flatpickr" */'flatpickr');
      import(/* webpackChunkName: "flatpickr-css" */'flatpickr/dist/flatpickr.min.css');
      const options = {
        ...ACMS.Config.flatDatePickerConfig
      };
      if (/^ja/.test(ACMS.i18n.lng)) {
        const lang = await import(/* webpackChunkName: "flatpickr-ja" */'flatpickr/dist/l10n/ja');
        options.locale = lang.Japanese;
      }
      [].forEach.call(flatPickerTarget, (item) => {
        if ($(item).hasClass('done')) {
          return;
        }
        options.defaultDate = item.value;
        const picker = flatPicker(item, options);
        item.setAttribute('autocomplete', 'off');
        item.addEventListener('input', (e) => {
          picker.jumpToDate(e.target.value);
          picker.setDate(e.target.value);
        });
        item.addEventListener('change', (e) => {
          picker.jumpToDate(e.target.value);
          picker.setDate(e.target.value);
        });
        $(item).addClass('done');
      });
    }

    //-------------------
    // timepicker
    const flatTimePickerTarget = document.querySelectorAll(ACMS.Config.flatTimePicker);
    if (flatTimePickerTarget && flatTimePickerTarget.length) {
      const { default: flatPicker } = await import(/* webpackChunkName: "flatpickr" */'flatpickr');
      await import(/* webpackChunkName: "flatpickr-css" */'flatpickr/dist/flatpickr.min.css');
      [].forEach.call(flatTimePickerTarget, (item) => {
        if ($(item).hasClass('done')) {
          return;
        }
        const picker = flatPicker(item, {
          ...ACMS.Config.flatTimePickerConfig,
          defaultDate: item.value,
        });
        item.setAttribute('autocomplete', 'off');
        item.addEventListener('input', (e) => {
          picker.jumpToDate(e.target.value);
          picker.setDate(e.target.value);
        });
        item.addEventListener('change', (e) => {
          picker.jumpToDate(e.target.value);
          picker.setDate(e.target.value);
        });
        $(item).addClass('done');
      });
    }
  };

  dispatchFlatPicker(document);
  ACMS.addListener('acmsAddCustomFieldGroup', (e) => {
    dispatchFlatPicker(e.target);
  });
  ACMS.addListener('acmsAddUnit', (e) => {
    dispatchFlatPicker(e.target);
  });

  //-------------------
  // contrast color
  const contrastColorTarget = document.querySelectorAll(ACMS.Config.contrastColorTarget);
  if (contrastColorTarget && contrastColorTarget.length) {
    [].forEach.call(contrastColorTarget, (item) => {
      const black = item.getAttribute('data-black-color') || '#000000';
      const white = item.getAttribute('data-white-color') || '#ffffff';
      let bgColor = item.getAttribute('data-bg-color');
      if (!bgColor) {
        const style = window.getComputedStyle(item);
        if (style) {
          bgColor = rgb2hex(style.backgroundColor);
        }
      }
      if (bgColor) {
        item.style.color = contrastColor(bgColor, black, white);
      }
    });
  }

  //-------------------
  // twitter login
  const twitterLogin = document.querySelector('.js-twitter-login');
  if (twitterLogin) {
    (async () => {
      const { default: login } = await import(/* webpackChunkName: "twitter-login" */'./lib/twitter-login');
      const type = twitterLogin.dataset.type;
      login(twitterLogin, type);
    })();
  }

  //-------------------
  // document-outliner
  const outlineTarget = document.querySelectorAll(ACMS.Config.documentOutlinerMark);
  if (outlineTarget && outlineTarget.length) {
    [].forEach.call(outlineTarget, (item) => {
      requestAnimationFrame(() => {
        const target = item.getAttribute('data-target');
        if (!target || !document.querySelector(target)) {
          return;
        }
        const outline = new DocumentOutliner(item);
        const overrideConfig = {};
        Object.keys(ACMS.Config.documentOutlinerConfig).forEach((key) => {
          let value = item.getAttribute(`data-${key}`);
          if (value) {
            if (isNaN(value) === false) {
              value = parseInt(value, 10);
            }
            if (value === 'true' || value === 'false') {
              value = value === 'true';
            }
            overrideConfig[key] = value;
          }
        });
        const config = Object.assign({}, ACMS.Config.documentOutlinerConfig, overrideConfig);

        outline.makeList(target, config);
        [].forEach.call(document.querySelectorAll(ACMS.Config.scrollToMark), (anchor) => {
          ACMS.Dispatch.scrollto(anchor);
        });
      });
    });
  }
};
