<?php

namespace Acms\Services\Update\System;

use Acms\Services\Facades\Storage;
use Symfony\Component\Finder\Finder;

/**
 * Class PlaceFile
 * @package Acms\Services\Update\System
 */
class PlaceFile
{
    /**
     * @var array
     */
    protected $moveList;

    /**
     * @var array
     */
    protected $exclusionMoveFile;

    /**
     * @var \Acms\Services\Update\Logger
     */
    protected $logger;

    /**
     * @var array
     */

    /**
     * PlaceFile constructor.
     *
     * @param \Acms\Services\Update\Logger $logger
     */
    public function __construct($logger)
    {
        set_time_limit(0);

        $this->logger = $logger;

        $this->moveList = array(
            'js' => 'js',
            'lang' => 'lang',
            'php' => 'php',
            'private/config.system.default.yaml' => 'private/config.system.default.yaml',
            'themes/system' => 'themes/system',
            'acms.js' => 'acms.js',
            'index.php' => 'index.php',
            'setup' => '_setup_' . date('YmdHi'),
        );

        $this->exclusionMoveFile = array_merge(configArray('system_update_ignore'), array(
            'php/AAPP',
            'php/ACMS/User',
        ));
    }

    /**
     * Validate
     *
     * @param $new_path
     * @param $backup_dir
     * @throws \Exception
     */
    public function validate($new_path, $backup_dir)
    {
        $this->logger->addMessage(gettext('アップデートの検証中...'), 0);
        $validate = true;
        // backup
        foreach ( $this->moveList as $item => $to ) {
            if ( $item === 'setup' ) {
                continue;
            }
            $path = $backup_dir . $item;
            Storage::makeDirectory(dirname($path));
            if (!Storage::isWritable(dirname($path))) {
                $validate = false;
                $this->logger->error(gettext('書き込み権限がありません。') . ' ' . $path);
            }
        }

        // place file
        foreach ( $this->moveList as $from => $to ) {
            if (!Storage::exists($to)) {
                $to = dirname($to);
            }
            if (!Storage::isWritable($to)) {
                $validate = false;
                $this->logger->error(gettext('書き込み権限がありません。') . ' ' . $to);
            }
        }
        foreach ( $this->exclusionMoveFile as $item ) {
            if (!Storage::isWritable($item)) {
                $validate = false;
                $this->logger->error(gettext('書き込み権限がありません。') . ' ' . $item);
            }
        }
        if (!$validate) {
            $this->logger->error(gettext('アップデートの検証に失敗しました。'));
            throw new \RuntimeException('');
        }
        sleep(5);
        $this->logger->addMessage(gettext('アップデートの検証完了'), 0);
    }

    /**
     * Run
     *
     * @param $new_path
     * @param $backup_dir
     * @param $new_setup
     * @throws \Exception
     */
    public function exec($new_path, $backup_dir, $new_setup = false)
    {
        try {
            if ($new_setup) {
                $this->removeSetup();
            } else {
                unset($this->moveList['setup']);
            }
            $this->backup($backup_dir);
            $this->updateFiles($new_path, $backup_dir);

        } catch ( \Exception $e ) {
            $this->logger->error($e->getMessage());
            $this->rollback($backup_dir);
            throw new \RuntimeException('');
        }
    }

    /**
     * System Update
     *
     * @param string $new_path
     * @param string $backup_dir
     */
    protected function updateFiles($new_path, $backup_dir)
    {
        $this->logger->addMessage(gettext('システムファイルを展開中...') . ' (' . $backup_dir . ')', 0);

        $base = $new_path . '/';

        $percentage = intval(25 / count($this->moveList));

        foreach ( $this->moveList as $from => $to ) {
            Storage::makeDirectory(dirname($backup_dir . 'tmp/' . $to));
            Storage::move($to, $backup_dir . 'tmp/' . $to);
            if ( !Storage::move($base . $from, $to) ) {
                throw new \RuntimeException('Could not be moved from ' . $base . $from . ' to ' . $to . '.');
            }
            $this->logger->addPercentage($percentage);
        }

        foreach ( $this->exclusionMoveFile as $item ) {
            if ( is_dir($item) && Storage::exists($backup_dir . $item) ) {
                if ( !Storage::copyDirectory($backup_dir . $item, $item) ) {
                    throw new \RuntimeException('Could not be copied from ' . $backup_dir . $item . ' to ' . $item . '.');
                }
            } else if (Storage::exists($backup_dir . $item) ) {
                if ( !Storage::copy($backup_dir . $item, $item) ) {
                    throw new \RuntimeException('Could not be copied from ' . $backup_dir . $item . ' to ' . $item . '.');
                }
            }
        }
        $this->logger->addMessage(gettext('システムファイルを展開完了'), 5);
    }

    /**
     * Backup
     *
     * @param string $backup_dir
     */
    protected function backup($backup_dir)
    {
        $this->logger->addMessage(gettext('バックアップ中...') . ' (' . $backup_dir . ')', 0);
        $backupList = $this->moveList;
        $backupList['config.server.php'] = 'config.server.php';

        foreach ( $backupList as $item => $to ) {
            if ( $item === 'setup' ) {
                continue;
            }
            Storage::makeDirectory(dirname($backup_dir . $item));
            $from = $item;
            if ( is_link($item) ) {
                if ( $link = readlink($item) ) {
                    $from = $link;
                }
            }
            if ( is_dir($item) && Storage::exists($from) ) {
                if ( !Storage::copyDirectory($from, $backup_dir . $item) ) {
                    throw new \RuntimeException('Could not be copied from ' . $from . ' to ' . $backup_dir . $item . '.');
                }
            } else if (Storage::exists($from)) {
                if ( !Storage::copy($from, $backup_dir . $item) ) {
                    throw new \RuntimeException('Could not be copied from ' . $from . ' to ' . $backup_dir . $item . '.');
                }
            }
        }
        $this->logger->addMessage(gettext('バックアップ完了'), 20);
    }

    /**
     * Rollback
     *
     * @param string $backup_dir
     */
    protected function rollback($backup_dir)
    {
        $this->logger->error(gettext('ロールバック中...'));

        foreach ( $this->moveList as $item => $to ) {
            if ( $item === 'setup' ) {
                continue;
            }
            try {
                Storage::makeDirectory(dirname($backup_dir . 'rollback/' . $to));
                Storage::move($to, $backup_dir . 'rollback/' . $to);
                Storage::move($backup_dir . $item, $item);
            } catch ( \Exception $e ) {
                $this->logger->error($e->getMessage());
            }
        }
        $this->logger->error(gettext('ロールバック終了'));
    }

    /**
     * Remove setup
     */
    protected function removeSetup()
    {
        $finder = new Finder();
        $lists = array();
        $iterator = $finder
            ->in('./')
            ->depth('< 2')
            ->name('/^\_setup\_.+/')
            ->directories();

        foreach ($iterator as $dir) {
            $lists[] = $dir->getRelativePathname();
        }
        foreach ($lists as $item) {
            try {
                Storage::removeDirectory($item);
            } catch (\Exception $e) {}
        }
    }
}
